<?php
// public/colaboradores/novo.php
require_once __DIR__ . '/../../app/core/auth.php';
require_login();

require_once __DIR__ . '/../../app/core/rbac.php';
require_role('admin'); // 🔒 somente admin

require_once __DIR__ . '/../../app/config/db.php';
require_once __DIR__ . '/../../app/core/helpers.php';

$pageTitle = 'Novo Colaborador';
include __DIR__ . '/../../app/partials/head.php';
include __DIR__ . '/../../app/partials/topbar.php';

/* Empresas ativas */
$empresas = $pdo->query("
  SELECT id, razao_social
    FROM empresas
   WHERE status='ativo'
   ORDER BY razao_social
")->fetchAll();

/* Estado do form / erros */
$errors = [];
$old = [
  'empresa_id' => $_POST['empresa_id'] ?? '',
  'nome'       => $_POST['nome']       ?? '',
  'cargo'      => $_POST['cargo']      ?? '',
  'email'      => $_POST['email']      ?? '',
];

/* POST */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $empresa_id = (int)$old['empresa_id'];
  $nome       = trim((string)$old['nome']);
  $cargo      = trim((string)$old['cargo']);
  $email      = trim((string)$old['email']);

  // Empresa válida e ativa?
  if ($empresa_id <= 0) {
    $errors[] = 'Selecione a empresa.';
  } else {
    $chkEmp = $pdo->prepare("SELECT COUNT(*) FROM empresas WHERE id=? AND status='ativo'");
    $chkEmp->execute([$empresa_id]);
    if (!(int)$chkEmp->fetchColumn()) {
      $errors[] = 'Empresa inválida ou inativa.';
    }
  }

  // Nome obrigatório
  if ($nome === '') {
    $errors[] = 'Nome é obrigatório.';
  }

  // E-mail opcional: se informado, valida formato e unicidade por empresa
  if ($email !== '') {
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
      $errors[] = 'E-mail inválido.';
    } else {
      $chk = $pdo->prepare("
        SELECT COUNT(*) FROM colaboradores
         WHERE email = ? AND empresa_id = ?
      ");
      $chk->execute([$email, $empresa_id]);
      if ((int)$chk->fetchColumn() > 0) {
        $errors[] = 'Já existe um colaborador com este e-mail nesta empresa.';
      }
    }
  } else {
    $email = null; // normaliza vazio
  }

  if (!$errors) {
    $ins = $pdo->prepare("
      INSERT INTO colaboradores (empresa_id, nome, cargo, email, status)
      VALUES (?, ?, ?, ?, 'ativo')
    ");
    $ok = $ins->execute([
      $empresa_id,
      $nome,
      $cargo !== '' ? $cargo : null,
      $email
    ]);

    if ($ok) {
      redirect('colaboradores/index.php');
    } else {
      $errors[] = 'Não foi possível salvar o colaborador.';
    }
  }
}
?>

<div class="d-flex justify-content-between align-items-center mb-3">
  <h1 class="h4 m-0">Novo colaborador</h1>
  <a href="<?= e(url('colaboradores/index.php')) ?>" class="btn btn-outline-secondary">Voltar</a>
</div>

<?php if ($errors): ?>
  <div class="alert alert-danger">
    <strong>Não foi possível salvar:</strong><br>
    <?= implode('<br>', array_map('e', $errors)) ?>
  </div>
<?php endif; ?>

<div class="card">
  <div class="card-body">
    <form method="post" class="row g-3">
      <div class="col-12 col-md-6">
        <label class="form-label">Empresa</label>
        <select name="empresa_id" class="form-select" required>
          <option value="">Selecione</option>
          <?php foreach ($empresas as $e): ?>
            <option value="<?= (int)$e['id'] ?>" <?= ((string)$old['empresa_id'] === (string)$e['id']) ? 'selected' : '' ?>>
              <?= e($e['razao_social']) ?>
            </option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="col-12 col-md-6">
        <label class="form-label">Nome</label>
        <input name="nome" class="form-control" required value="<?= e($old['nome']) ?>">
      </div>

      <div class="col-12 col-md-6">
        <label class="form-label">Cargo <span class="text-muted">(opcional)</span></label>
        <input name="cargo" class="form-control" value="<?= e($old['cargo']) ?>">
      </div>

      <div class="col-12 col-md-6">
        <label class="form-label">E-mail <span class="text-muted">(opcional)</span></label>
        <input type="email" name="email" class="form-control" value="<?= e($old['email']) ?>">
      </div>

      <div class="col-12">
        <button class="btn btn-primary">Salvar</button>
        <a href="<?= e(url('colaboradores/index.php')) ?>" class="btn btn-outline-secondary">Cancelar</a>
      </div>
    </form>
  </div>
</div>

<?php include __DIR__ . '/../../app/partials/foot.php'; ?>
