<?php
// public/comparador/temporal.php
require_once __DIR__ . '/../../app/core/auth.php';
require_login();

require_once __DIR__ . '/../../app/core/rbac.php';
require_role(['admin']); // só admin aqui; ajuste se quiser

require_once __DIR__ . '/../../app/config/db.php';
require_once __DIR__ . '/../../app/core/helpers.php';

$pageTitle = 'Comparador Temporal por Gerência';
include __DIR__ . '/../../app/partials/head.php';
include __DIR__ . '/../../app/partials/topbar.php';

/* ---------------- Parâmetros ---------------- */
$empresaId   = (int)($_GET['empresa_id'] ?? 0);
$checklistId = (int)($_GET['checklist_id'] ?? 0);
$de          = trim($_GET['de'] ?? '');
$ate         = trim($_GET['ate'] ?? '');
$modo        = $_GET['modo'] ?? 'todas'; // 'todas' | 'intersec'

function qs_keep(array $extra = []): string {
  $cur = $_GET;
  foreach ($extra as $k=>$v) {
    if ($v === null) unset($cur[$k]); else $cur[$k] = $v;
  }
  return http_build_query($cur);
}

/* ---------------- Opções (empresa, checklist) ---------------- */
$empresas = $pdo->query("SELECT id, razao_social FROM empresas ORDER BY razao_social")->fetchAll(PDO::FETCH_ASSOC);

$checklists = [];
if ($empresaId > 0) {
  $st = $pdo->prepare("
    SELECT id, titulo
      FROM checklists
     WHERE empresa_id = ?
     ORDER BY titulo
  ");
  $st->execute([$empresaId]);
  $checklists = $st->fetchAll(PDO::FETCH_ASSOC);
}

/* ---------------- Buscar execuções finalizadas (linhas) ----------------
   Vamos carregar um dataset “alto nível” por execução, gerência, pergunta_uid
   para permitir os dois modos de cálculo. */
$linhas = []; // []: cada item -> [execucao_id, finalizado_em, gerencia_id, gerencia, pergunta_uid, peso, resposta]
$execDatas = []; // exec_id => 'YYYY-mm-dd HH:ii:ss'
if ($empresaId > 0 && $checklistId > 0) {
  $where = " e.status='finalizado' AND e.empresa_id=:emp AND e.checklist_id=:chk ";
  $params = [':emp'=>$empresaId, ':chk'=>$checklistId];

  if ($de !== '') { $where .= " AND e.finalizado_em >= :de ";  $params[':de']  = $de.' 00:00:00'; }
  if ($ate !== ''){ $where .= " AND e.finalizado_em <= :ate "; $params[':ate'] = $ate.' 23:59:59'; }

  $sql = $pdo->prepare("
    SELECT
      e.id AS execucao_id, e.finalizado_em,
      g.id AS gerencia_id, g.nome AS gerencia,
      p.pergunta_uid, p.peso,
      r.resposta
    FROM execucoes e
    JOIN respostas r              ON r.execucao_id = e.id
    JOIN checklist_perguntas p    ON p.id = r.pergunta_id
    LEFT JOIN gerencias g         ON g.id = p.gerencia_id
    WHERE $where
    ORDER BY e.finalizado_em ASC, e.id ASC, g.nome ASC, p.pergunta_uid ASC
  ");
  $sql->execute($params);
  while ($row = $sql->fetch(PDO::FETCH_ASSOC)) {
    $linhas[] = [
      'execucao_id'  => (int)$row['execucao_id'],
      'finalizado_em'=> $row['finalizado_em'],
      'gerencia_id'  => (int)($row['gerencia_id'] ?? 0),
      'gerencia'     => $row['gerencia'] ?? '—',
      'pergunta_uid' => $row['pergunta_uid'] ?? null,
      'peso'         => (float)$row['peso'],
      'resposta'     => $row['resposta'],
    ];
    $execDatas[(int)$row['execucao_id']] = $row['finalizado_em'];
  }
}

/* ---------------- Preparar Execuções (eixo do tempo) ---------------- */
$execsOrdenadas = [];
if ($execDatas) {
  // ordena por data, depois id
  asort($execDatas);
  foreach ($execDatas as $eid=>$dt) $execsOrdenadas[] = (int)$eid;
}

/* ---------------- Cálculo por gerência (dois modos) ---------------- */
$gerLabels = []; // gid => nome
$seriesPorGerencia = []; // gid => [ [eid=>percent], ... ] (iremos montar em forma de datasets para Chart.js)
$tabelaGer = []; // gid => [eid=>percent]

if ($linhas && $execsOrdenadas) {
  // 1) agrupar base: por (exec -> gerencia -> pergunta_uid)
  $base = []; // [eid][gid]['gerencia'=>..., 'itens'][] = ['uid'=>..., 'peso'=>..., 'val'=>0|1|null]
  foreach ($linhas as $r) {
    $eid = (int)$r['execucao_id'];
    $gid = (int)$r['gerencia_id'];
    $gno = $r['gerencia'] ?: '—';
    $uid = $r['pergunta_uid'];
    $peso= (float)$r['peso'];
    $val = ($r['resposta']==='conforme') ? 1 : (($r['resposta']==='nao_conforme') ? 0 : null);

    if (!isset($base[$eid])) $base[$eid] = [];
    if (!isset($base[$eid][$gid])) $base[$eid][$gid] = ['gerencia'=>$gno, 'itens'=>[]];
    $base[$eid][$gid]['itens'][] = ['uid'=>$uid, 'peso'=>$peso, 'val'=>$val];
    $gerLabels[$gid] = $gno;
  }

  // 2) se modo == intersec, calcular interseção de pergunta_uid por gerência ao longo de TODAS execuções
  $intersec = []; // gid => set(uids)
  if ($modo === 'intersec') {
    // start: interseção = null; primeira exec define o conjunto, as seguintes intersectam
    foreach ($execsOrdenadas as $eid) {
      $gers = $base[$eid] ?? [];
      foreach ($gers as $gid=>$info) {
        $uids = [];
        foreach ($info['itens'] as $it) if ($it['uid']) $uids[$it['uid']] = true;
        $setE = array_keys($uids);

        if (!isset($intersec[$gid])) {
          $intersec[$gid] = array_fill_keys($setE, true);
        } else {
          // intersect
          $cur = $intersec[$gid];
          $new = [];
          foreach ($setE as $u) { if (isset($cur[$u])) $new[$u] = true; }
          $intersec[$gid] = $new;
        }
      }
    }
  }

  // 3) calcular percentuais por gerência e execução
  foreach ($execsOrdenadas as $eid) {
    $gers = $base[$eid] ?? [];
    foreach ($gerLabels as $gid=>$gno) {
      $itens = $gers[$gid]['itens'] ?? [];

      // se intersec: filtra por uids em intersec[gid]
      if ($modo === 'intersec') {
        $keep = $intersec[$gid] ?? [];
        $itens = array_values(array_filter($itens, function($it) use ($keep){
          if (!$it['uid']) return false; // sem uid não é comparável
          return isset($keep[$it['uid']]);
        }));
      }

      $w=0.0; $v=0.0;
      foreach ($itens as $it) {
        if ($it['val'] === null || $it['peso'] <= 0) continue;
        $w += (float)$it['peso'];
        $v += (float)$it['peso'] * (int)$it['val'];
      }
      $perc = $w>0 ? round(($v/$w)*100, 2) : null;

      $tabelaGer[$gid][$eid] = $perc;
    }
  }

  // 4) montar séries para Chart.js (cada gerência vira uma linha)
  // labels do eixo X = datas das execuções
  // data = % em cada execução, na ordem de $execsOrdenadas
  $seriesPorGerencia = []; // gid => ['label'=>..., 'data'=>[...]]
  foreach ($gerLabels as $gid=>$gno) {
    $serie = [];
    foreach ($execsOrdenadas as $eid) {
      $serie[] = isset($tabelaGer[$gid][$eid]) && $tabelaGer[$gid][$eid] !== null ? (float)$tabelaGer[$gid][$eid] : null;
    }
    $seriesPorGerencia[$gid] = ['label'=>$gno, 'data'=>$serie];
  }
}

/* Paleta de cores para gerências */
$palette = [
  '#0d6efd','#198754','#dc3545','#fd7e14','#20c997',
  '#6f42c1','#0dcaf0','#ffc107','#6610f2','#6c757d',
  '#d63384','#1984a0','#7d9c00','#8e44ad','#2c3e50'
];
?>
<style>
  .filters .form-select, .filters .form-control { min-width: 220px; }
  .card-section { border-radius: 1rem; overflow: hidden; }
  .muted { color: #6c757d; }
  .table-fixed th, .table-fixed td { white-space: nowrap; }
</style>

<div class="container my-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h1 class="h4 m-0">Comparador Temporal por Gerência</h1>
    <div class="d-flex gap-2">
      <a class="btn btn-outline-secondary" href="<?= e(url('comparador/index.php')) ?>">Comparador (Execuções)</a>
    </div>
  </div>

  <!-- Filtros -->
  <form method="get" class="row gy-2 gx-2 align-items-end filters mb-3">
    <div class="col-12 col-md-3">
      <label class="form-label mb-1">Empresa</label>
      <select name="empresa_id" class="form-select" onchange="this.form.submit()">
        <option value="0">Selecione…</option>
        <?php foreach ($empresas as $e): ?>
          <option value="<?= (int)$e['id'] ?>" <?= $empresaId==$e['id']?'selected':'' ?>><?= e($e['razao_social']) ?></option>
        <?php endforeach; ?>
      </select>
    </div>

    <div class="col-12 col-md-4">
      <label class="form-label mb-1">Checklist</label>
      <select name="checklist_id" class="form-select" <?= $empresaId>0?'':'disabled' ?> onchange="this.form.submit()">
        <option value="0"><?= $empresaId>0 ? 'Selecione…' : 'Escolha a empresa' ?></option>
        <?php foreach ($checklists as $c): ?>
          <option value="<?= (int)$c['id'] ?>" <?= $checklistId==$c['id']?'selected':'' ?>><?= e($c['titulo']) ?></option>
        <?php endforeach; ?>
      </select>
    </div>

    <div class="col-6 col-md-2">
      <label class="form-label mb-1">De</label>
      <input type="date" class="form-control" name="de" value="<?= e($de) ?>">
    </div>
    <div class="col-6 col-md-2">
      <label class="form-label mb-1">Até</label>
      <input type="date" class="form-control" name="ate" value="<?= e($ate) ?>">
    </div>

    <div class="col-12 col-md-3">
      <label class="form-label mb-1">Modo</label>
      <select name="modo" class="form-select">
        <option value="todas"   <?= $modo==='todas'?'selected':'' ?>>Todas as perguntas presentes</option>
        <option value="intersec"<?= $modo==='intersec'?'selected':'' ?>>Somente interseção por pergunta_uid</option>
      </select>
    </div>

    <div class="col-12">
      <button class="btn btn-primary">Aplicar</button>
      <a class="btn btn-outline-secondary" href="<?= e(strtok($_SERVER['REQUEST_URI'], '?')) ?>">Limpar</a>
    </div>
  </form>

  <?php if ($empresaId === 0 || $checklistId === 0): ?>
    <div class="alert alert-info">Selecione empresa e checklist para visualizar.</div>
  <?php elseif (!$linhas): ?>
    <div class="alert alert-warning">Nenhuma execução finalizada encontrada para o período/ filtros informados.</div>
  <?php else: ?>
    <div class="card card-section mb-4">
      <div class="card-body">
        <h5 class="card-title">Série temporal por Gerência (<?= $modo==='intersec' ? 'interseção por pergunta_uid' : 'todas as perguntas presentes' ?>)</h5>
        <canvas id="chartTemporal" height="160" aria-label="Série temporal por Gerência"></canvas>

        <!-- Tabela -->
        <div class="table-responsive mt-3">
          <table class="table table-sm table-striped table-fixed align-middle">
            <thead>
              <tr>
                <th>Gerência \ Execução</th>
                <?php foreach ($execsOrdenadas as $eid): ?>
                  <th><?= 'Exec #'.(int)$eid ?><br><small class="muted"><?= e(date('d/m/Y H:i', strtotime($execDatas[$eid]))) ?></small></th>
                <?php endforeach; ?>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($gerLabels as $gid=>$gno): ?>
                <tr>
                  <td><strong><?= e($gno ?: '—') ?></strong></td>
                  <?php foreach ($execsOrdenadas as $eid): $v = $tabelaGer[$gid][$eid] ?? null; ?>
                    <td><?= $v!==null ? e(number_format($v,2,',','.')).'%' : '—' ?></td>
                  <?php endforeach; ?>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>

        <div class="small muted mt-2">
          * Cálculo: (∑ peso×valor / ∑ peso)×100, valor=1 (conforme), 0 (não conforme).
          <?= $modo==='intersec' ? ' | Considerando apenas pergunta_uid presente em TODAS as execuções do período.' : '' ?>
        </div>
      </div>
    </div>
  <?php endif; ?>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
(function(){
  const execIds   = <?= json_encode($execsOrdenadas) ?>;
  const execDates = <?= json_encode(array_map(fn($eid)=> date('d/m/Y', strtotime($execDatas[$eid] ?? '')), $execsOrdenadas), JSON_UNESCAPED_UNICODE) ?>;
  const series    = <?= json_encode($seriesPorGerencia, JSON_UNESCAPED_UNICODE) ?>; // {gid:{label,data[]}}
  const palette   = <?= json_encode($palette) ?>;

  function colorFor(idx){ return palette[idx % palette.length]; }

  if (execIds.length && Object.keys(series).length) {
    const ctx = document.getElementById('chartTemporal');
    if (!ctx) return;

    const datasets = [];
    let i = 0;
    Object.values(series).forEach(s => {
      datasets.push({
        label: s.label,
        data: s.data.map(v => v===null ? null : Number(v)),
        borderColor: colorFor(i),
        backgroundColor: colorFor(i),
        tension: 0.25,
        spanGaps: true
      });
      i++;
    });

    new Chart(ctx, {
      type: 'line',
      data: { labels: execDates, datasets },
      options: {
        responsive: true,
        scales: {
          y: { beginAtZero: true, max: 100, ticks: { callback: (v)=> v+'%' } }
        },
        plugins: {
          tooltip: {
            callbacks: {
              label: (ctx)=>{
                const v = ctx.parsed?.y ?? null;
                return (ctx.dataset?.label || '') + ': ' + (v!==null ? v.toFixed(2)+'%' : '—');
              }
            }
          },
          legend: { position: 'bottom' }
        }
      }
    });
  }
})();
</script>

<?php include __DIR__ . '/../../app/partials/foot.php'; ?>
