<?php
// public/empresa_user/index.php
require_once __DIR__ . '/../../app/core/auth.php';
require_login();

$u = current_user();
if (($u['role'] ?? '') !== 'empresa_user') {
  redirect('../login.php');
}

require_once __DIR__ . '/../../app/config/db.php';
require_once __DIR__ . '/../../app/core/helpers.php';

$pageTitle = 'Minhas Execuções';

$userId    = (int)($u['id'] ?? 0);
$empresaId = (int)($u['empresa_id'] ?? 0);

if (!$empresaId) {
  include __DIR__ . '/../../app/partials/head.php';
  include __DIR__ . '/../../app/partials/topbar.php';
  ?>
  <div class="container mt-4">
    <h1 class="h4 mb-3">Execuções</h1>
    <div class="alert alert-warning">
      Seu usuário não está vinculado a nenhuma empresa. Entre em contato com o administrador.
    </div>
  </div>
  <?php
  include __DIR__ . '/../../app/partials/foot.php';
  exit;
}

// Nome da empresa (opcional)
$empresaNome = "Empresa #{$empresaId}";
try {
  $qEmp = $pdo->prepare("SELECT COALESCE(razao_social, CONCAT('Empresa #', id)) FROM empresas WHERE id=? LIMIT 1");
  $qEmp->execute([$empresaId]);
  $empresaNome = $qEmp->fetchColumn() ?: $empresaNome;
} catch (Throwable $e) { /* ignore */ }

// -----------------------------
// Filtros & paginação
// -----------------------------
$dini   = trim($_GET['dini'] ?? '');
$dfim   = trim($_GET['dfim'] ?? '');
$diniDT = $dini ? ($dini . ' 00:00:00') : null;
$dfimDT = $dfim ? ($dfim . ' 23:59:59') : null;

$ppDisp = max(3, min(30, (int)($_GET['ppd'] ?? 9)));
$ppHist = max(3, min(30, (int)($_GET['pph'] ?? 9)));
$pDisp  = max(1, (int)($_GET['pd']  ?? 1));
$pHist  = max(1, (int)($_GET['ph']  ?? 1));
$offDisp = ($pDisp - 1) * $ppDisp;
$offHist = ($pHist - 1) * $ppHist;

// -----------------------------
// DISPONÍVEIS
// Regra:
//  - Mostrar execuções da MESMA EMPRESA com status 'liberada'
//  - Mostrar execuções 'em_andamento' SOMENTE se executado_por = usuário atual
// -----------------------------
$sqlCountDisp = $pdo->prepare("
  SELECT COUNT(*) 
    FROM execucoes x
   WHERE x.empresa_id = ?
     AND (
           x.status = 'liberada'
        OR (x.status = 'em_andamento' AND x.executado_por = ?)
     )
");
$sqlCountDisp->execute([$empresaId, $userId]);
$totalDisp = (int)$sqlCountDisp->fetchColumn();
$totalPagDisp = max(1, (int)ceil($totalDisp / $ppDisp));

$sqlDisp = $pdo->prepare("
  SELECT x.id, x.status, x.iniciado_em, x.finalizado_em, x.executado_por,
         c.titulo AS checklist_titulo
    FROM execucoes x
    JOIN checklists c ON c.id = x.checklist_id
   WHERE x.empresa_id = :emp
     AND (
           x.status = 'liberada'
        OR (x.status = 'em_andamento' AND x.executado_por = :uid)
     )
ORDER BY
  CASE x.status WHEN 'em_andamento' THEN 0 WHEN 'liberada' THEN 1 ELSE 9 END,
  x.id DESC
LIMIT :lim OFFSET :off
");
$sqlDisp->bindValue(':emp', $empresaId, PDO::PARAM_INT);
$sqlDisp->bindValue(':uid', $userId, PDO::PARAM_INT);
$sqlDisp->bindValue(':lim', $ppDisp, PDO::PARAM_INT);
$sqlDisp->bindValue(':off', $offDisp, PDO::PARAM_INT);
$sqlDisp->execute();
$disponiveis = $sqlDisp->fetchAll(PDO::FETCH_ASSOC);

// -----------------------------
// HISTÓRICO (apenas do próprio usuário) 
// -----------------------------
$whereHist = "x.empresa_id = :emp AND x.status = 'finalizado' AND x.executado_por = :uid";
$paramsHist = [':emp' => $empresaId, ':uid' => $userId];
if ($diniDT) { $whereHist .= " AND x.finalizado_em >= :dini"; $paramsHist[':dini'] = $diniDT; }
if ($dfimDT) { $whereHist .= " AND x.finalizado_em <= :dfim"; $paramsHist[':dfim'] = $dfimDT; }

$sqlCountHist = $pdo->prepare("SELECT COUNT(*) FROM execucoes x WHERE $whereHist");
$sqlCountHist->execute($paramsHist);
$totalHist = (int)$sqlCountHist->fetchColumn();
$totalPagHist = max(1, (int)ceil($totalHist / $ppHist));

$sqlHist = $pdo->prepare("
  SELECT x.id, x.finalizado_em, c.titulo AS checklist_titulo
    FROM execucoes x
    JOIN checklists c ON c.id = x.checklist_id
   WHERE $whereHist
ORDER BY x.finalizado_em DESC, x.id DESC
LIMIT :lim OFFSET :off
");
foreach ($paramsHist as $k=>$v) $sqlHist->bindValue($k, $v);
$sqlHist->bindValue(':lim', $ppHist, PDO::PARAM_INT);
$sqlHist->bindValue(':off', $offHist, PDO::PARAM_INT);
$sqlHist->execute();
$historico = $sqlHist->fetchAll(PDO::FETCH_ASSOC);

// -----------------------------
// Helpers de paginação
// -----------------------------
function build_qs(array $extra = []): string {
  $cur = $_GET;
  foreach ($extra as $k=>$v) {
    if ($v === null) unset($cur[$k]); else $cur[$k] = $v;
  }
  return http_build_query($cur);
}
function render_pagination_simple(int $current, int $total, string $param) {
  if ($total <= 1) return;
  $base = strtok($_SERVER['REQUEST_URI'], '?');
  $window = 3;
  $ini = max(1, $current - 1);
  $fim = min($total, $ini + $window - 1);
  if ($fim - $ini + 1 < $window) $ini = max(1, $fim - $window + 1);

  echo '<nav><ul class="pagination justify-content-center">';
  $prev = max(1, $current - 1);
  echo '<li class="page-item'.($current<=1?' disabled':'').'"><a class="page-link" href="'.e($base.'?'.build_qs([$param=>$prev])).'">Anterior</a></li>';
  for($i=$ini;$i<=$fim;$i++){
    echo '<li class="page-item'.($i===$current?' active':'').'"><a class="page-link" href="'.e($base.'?'.build_qs([$param=>$i])).'">'.(int)$i.'</a></li>';
  }
  $next = min($total, $current + 1);
  echo '<li class="page-item'.($current>=$total?' disabled':'').'"><a class="page-link" href="'.e($base.'?'.build_qs([$param=>$next])).'">Próxima</a></li>';
  echo '</ul></nav>';
}

include __DIR__ . '/../../app/partials/head.php';
include __DIR__ . '/../../app/partials/topbar.php';
?>
<style>
  .card-check{border:1px solid #e9ecef; border-radius:1rem; overflow:hidden; transition:transform .08s ease, box-shadow .12s ease;}
  .card-check:hover{transform:translateY(-2px); box-shadow:0 6px 24px rgba(0,0,0,.06)}
  .muted{color:#6c757d}
  .badge-status{text-transform:uppercase}
</style>

<div class="container my-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h1 class="h4 m-0">Execuções</h1>
    <span class="badge bg-secondary"><?= e($empresaNome) ?></span>
  </div>

  <?php if (!empty($_GET['debug'])): ?>
    <div class="alert alert-secondary">
      <div><strong>Debug</strong></div>
      <div>empresa_id do usuário: <?= (int)$empresaId ?></div>
      <div>user_id: <?= (int)$userId ?></div>
      <div>Total disponíveis (contagem): <?= (int)$totalDisp ?></div>
      <div>Total histórico (contagem): <?= (int)$totalHist ?></div>
    </div>
  <?php endif; ?>

  <!-- Disponíveis -->
  <div class="d-flex justify-content-between align-items-center mb-2">
    <h2 class="h5 m-0">Disponíveis para responder</h2>
    <span class="badge bg-primary"><?= (int)$totalDisp ?></span>
  </div>

  <?php if (!$disponiveis): ?>
    <div class="alert alert-info">Nenhuma execução liberada para você no momento.</div>
  <?php else: ?>
    <div class="row g-3 mb-3">
      <?php foreach ($disponiveis as $x): ?>
        <div class="col-12 col-md-6 col-lg-4">
          <div class="card card-check h-100">
            <div class="card-header d-flex justify-content-between align-items-center">
              <span class="fw-semibold"><?= e($x['checklist_titulo']) ?></span>
              <?php
                $st = (string)$x['status'];
                $cls = $st==='em_andamento' ? 'bg-warning text-dark' : ($st==='liberada' ? 'bg-primary' : 'bg-secondary');
              ?>
              <span class="badge badge-status <?= $cls ?>"><?= e($st) ?></span>
            </div>
            <div class="card-body d-flex flex-column">
              <ul class="list-unstyled muted small mb-3">
                <li>ID Execução: <strong>#<?= (int)$x['id'] ?></strong></li>
                <li>Iniciado: <strong><?= e($x['iniciado_em'] ?: '—') ?></strong></li>
              </ul>
              <div class="mt-auto d-grid">
                <a class="btn btn-primary" href="<?= e(url('empresa_user/responder.php?execucao_id='.(int)$x['id'])) ?>">
                  <?= $x['status']==='liberada' ? 'Responder agora' : 'Continuar' ?>
                </a>
              </div>
            </div>
          </div>
        </div>
      <?php endforeach; ?>
    </div>
    <?php render_pagination_simple($pDisp, $totalPagDisp, 'pd'); ?>
  <?php endif; ?>

  <!-- Histórico (do próprio usuário) -->
  <div class="d-flex justify-content-between align-items-center mt-4 mb-2">
    <h2 class="h5 m-0">Histórico (minhas finalizadas)</h2>
    <form class="d-flex align-items-end gap-2 m-0" method="get">
      <div>
        <label class="form-label mb-1">De</label>
        <input type="date" class="form-control" name="dini" value="<?= e($dini) ?>">
      </div>
      <div>
        <label class="form-label mb-1">Até</label>
        <input type="date" class="form-control" name="dfim" value="<?= e($dfim) ?>">
      </div>
      <input type="hidden" name="ppd" value="<?= (int)$ppDisp ?>">
      <input type="hidden" name="pph" value="<?= (int)$ppHist ?>">
      <input type="hidden" name="pd" value="<?= (int)$pDisp ?>">
      <button class="btn btn-outline-secondary">Filtrar histórico</button>
      <a class="btn btn-link" href="<?= e(strtok($_SERVER['REQUEST_URI'], '?')) ?>">Limpar</a>
    </form>
  </div>

  <?php if (!$historico): ?>
    <div class="alert alert-light border">Nenhuma execução finalizada sua no período.</div>
  <?php else: ?>
    <div class="table-responsive">
      <table class="table table-sm table-striped align-middle">
        <thead>
          <tr>
            <th style="width:90px">#</th>
            <th>Checklist</th>
            <th style="width:220px">Finalizado em</th>
            <th style="width:120px"></th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($historico as $h): ?>
            <tr>
              <td>#<?= (int)$h['id'] ?></td>
              <td><?= e($h['checklist_titulo']) ?></td>
              <td><?= e($h['finalizado_em'] ?: '—') ?></td>
              <td class="text-end">
                <a class="btn btn-sm btn-outline-secondary" href="<?= e(url('execucoes/detalhe.php?id='.(int)$h['id'])) ?>">Detalhe</a>
              </td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
    <?php render_pagination_simple($pHist, $totalPagHist, 'ph'); ?>
  <?php endif; ?>
</div>

<?php include __DIR__ . '/../../app/partials/foot.php'; ?>
