<?php
// public/setores/editar.php
require_once __DIR__ . '/../../app/core/auth.php';
require_login();

require_once __DIR__ . '/../../app/core/rbac.php';
require_role('admin'); // 🔒 somente admin

require_once __DIR__ . '/../../app/config/db.php';
require_once __DIR__ . '/../../app/core/helpers.php';

$pageTitle = 'Editar Setor';

/* Carregar registro alvo */
$id = (int)($_GET['id'] ?? 0);
$st = $pdo->prepare("SELECT * FROM setores WHERE id=?");
$st->execute([$id]);
$r = $st->fetch();

if (!$r) {
  include __DIR__ . '/../../app/partials/head.php';
  include __DIR__ . '/../../app/partials/topbar.php';
  echo "<div class='container mt-4'><div class='alert alert-danger'>Registro não encontrado.</div></div>";
  include __DIR__ . '/../../app/partials/foot.php';
  exit;
}

/* Empresas ativas para o select */
$empresas = $pdo->query("
  SELECT id, razao_social
    FROM empresas
   WHERE status='ativo'
   ORDER BY razao_social
")->fetchAll();

/* Estado do form (mantém valores em caso de erro) */
$errors = [];
$old = [
  'empresa_id'  => $_POST['empresa_id']  ?? $r['empresa_id'],
  'gerencia_id' => $_POST['gerencia_id'] ?? $r['gerencia_id'],
  'nome'        => $_POST['nome']        ?? $r['nome'],
  'observacao'  => $_POST['observacao']  ?? ($r['observacao'] ?? ''),
  'status'      => $_POST['status']      ?? $r['status'],
];

/* POST: salvar */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $empresa_id  = (int)($_POST['empresa_id'] ?? 0);
  $gerencia_id = (int)($_POST['gerencia_id'] ?? 0);
  $nome        = trim($_POST['nome'] ?? '');
  $observacao  = trim($_POST['observacao'] ?? '');
  $status      = $_POST['status'] ?? 'ativo';

  // validações
  if ($empresa_id <= 0) {
    $errors[] = 'Selecione a empresa.';
  } else {
    $chkEmp = $pdo->prepare("SELECT COUNT(*) FROM empresas WHERE id=? AND status='ativo'");
    $chkEmp->execute([$empresa_id]);
    if (!(int)$chkEmp->fetchColumn()) {
      $errors[] = 'Empresa inválida ou inativa.';
    }
  }

  if ($gerencia_id <= 0) {
    $errors[] = 'Selecione a gerência.';
  } else {
    // a gerência precisa existir, estar ativa e pertencer à MESMA empresa escolhida
    $chkGer = $pdo->prepare("SELECT COUNT(*) FROM gerencias WHERE id=? AND empresa_id=? AND status='ativo'");
    $chkGer->execute([$gerencia_id, $empresa_id]);
    if (!(int)$chkGer->fetchColumn()) {
      $errors[] = 'Gerência inválida para a empresa selecionada.';
    }
  }

  if ($nome === '') {
    $errors[] = 'Nome do Setor é obrigatório.';
  }

  if (!in_array($status, ['ativo','inativo'], true)) {
    $errors[] = 'Status inválido.';
  }

  if (!$errors) {
    $up = $pdo->prepare("
      UPDATE setores
         SET empresa_id = ?, gerencia_id = ?, nome = ?, observacao = ?, status = ?
       WHERE id = ?
    ");
    $ok = $up->execute([$empresa_id, $gerencia_id, $nome, $observacao !== '' ? $observacao : null, $status, $id]);

    if ($ok) {
      // Redireciona com flag de sucesso
      redirect('setores/editar.php?id='.$id.'&ok=1');
    } else {
      $errors[] = 'Não foi possível salvar as alterações.';
    }
  }
}

/* View */
include __DIR__ . '/../../app/partials/head.php';
include __DIR__ . '/../../app/partials/topbar.php';
?>

<div class="container mt-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h1 class="h4 m-0">Editar setor</h1>
    <div class="d-flex gap-2">
      <span class="badge bg-secondary align-self-center">ID: <?= (int)$id ?></span>
      <a href="<?= e(url('setores/index.php')) ?>" class="btn btn-outline-secondary btn-sm">Voltar</a>
    </div>
  </div>

  <?php if (isset($_GET['ok'])): ?>
    <div class="alert alert-success">Setor atualizado com sucesso.</div>
  <?php endif; ?>

  <?php if ($errors): ?>
    <div class="alert alert-danger">
      <strong>Não foi possível salvar:</strong><br>
      <?= implode('<br>', array_map('e', $errors)) ?>
    </div>
  <?php endif; ?>

  <div class="card">
    <div class="card-body">
      <form method="post" class="row g-3">
        <div class="col-12 col-md-6">
          <label class="form-label">Empresa</label>
          <select name="empresa_id" id="empresa" class="form-select" required>
            <?php foreach ($empresas as $e): ?>
              <option value="<?= (int)$e['id'] ?>"
                <?= ((string)$old['empresa_id'] === (string)$e['id']) ? 'selected' : '' ?>>
                <?= e($e['razao_social']) ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="col-12 col-md-6">
          <label class="form-label">Gerência</label>
          <select name="gerencia_id" id="gerencia" class="form-select" required>
            <!-- opções carregadas via JS conforme a empresa -->
          </select>
        </div>

        <div class="col-12">
          <label class="form-label">Nome</label>
          <input name="nome" class="form-control" value="<?= e($old['nome']) ?>" required>
        </div>

        <div class="col-12">
          <label class="form-label">Observação <span class="text-muted">(opcional)</span></label>
          <input name="observacao" class="form-control" value="<?= e($old['observacao']) ?>">
        </div>

        <div class="col-12 col-md-3">
          <label class="form-label">Status</label>
          <select name="status" class="form-select">
            <option value="ativo"   <?= $old['status']==='ativo'?'selected':'' ?>>Ativo</option>
            <option value="inativo" <?= $old['status']==='inativo'?'selected':'' ?>>Inativo</option>
          </select>
        </div>

        <div class="col-12">
          <button class="btn btn-primary">Salvar</button>
          <a href="<?= e(url('setores/index.php')) ?>" class="btn btn-outline-secondary">Cancelar</a>
        </div>
      </form>
    </div>
  </div>
</div>

<script>
async function carregarGerencias(empresaId, selectedId=0){
  const gSel = document.getElementById('gerencia');
  gSel.innerHTML = '<option value="">Carregando...</option>';
  if(!empresaId){
    gSel.innerHTML = '<option value="">Selecione a empresa</option>';
    return;
  }
  try {
    const res = await fetch('<?= e(url('api/gerencias_by_empresa.php')) ?>?empresa_id='+encodeURIComponent(empresaId));
    const js = await res.json();
    gSel.innerHTML = js.length ? '' : '<option value="">Nenhuma gerência encontrada</option>';
    js.forEach(g=>{
      const opt = document.createElement('option');
      opt.value = g.id; opt.textContent = g.nome;
      if(Number(selectedId)===Number(g.id)) opt.selected = true;
      gSel.appendChild(opt);
    });
  } catch(e) {
    gSel.innerHTML = '<option value="">Erro ao carregar</option>';
  }
}

window.addEventListener('DOMContentLoaded', ()=>{
  // carga inicial com valores atuais
  carregarGerencias('<?= (int)$old['empresa_id'] ?>','<?= (int)$old['gerencia_id'] ?>');

  // quando trocar a empresa, recarrega gerências e limpa seleção
  document.getElementById('empresa').addEventListener('change', (ev)=>{
    carregarGerencias(ev.target.value, 0);
  });
});
</script>

<?php include __DIR__ . '/../../app/partials/foot.php'; ?>
